<?php

require_once 'config.php';
require_once 'NameGenerator.php';

class SheerIDVerifier
{
    private $verificationId;
    private $deviceFingerprint;
    private $cookieFile; // File to store cookies

    public function __construct($verificationId)
    {
        $this->verificationId = $verificationId;
        $this->deviceFingerprint = $this->generateDeviceFingerprint();
        // Create a unique temp file for cookies per instance
        $this->cookieFile = tempnam(sys_get_temp_dir(), 'shearid_cookie_');
    }

    public function __destruct()
    {
        // Cleanup cookie file
        if (file_exists($this->cookieFile)) {
            unlink($this->cookieFile);
        }
    }

    private function generateDeviceFingerprint()
    {
        $chars = '0123456789abcdef';
        $result = '';
        for ($i = 0; $i < 32; $i++) {
            $result .= $chars[rand(0, 15)];
        }
        return $result;
    }

    public static function parseVerificationId($url)
    {
        if (preg_match('/verificationId=([a-f0-9]+)/i', $url, $matches)) {
            return $matches[1];
        }
        return null;
    }

    private function sheeridRequest($method, $url, $body = null)
    {
        $ch = curl_init();

        $headers = [
            "Content-Type: application/json",
            "User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36"
        ];

        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true); // Follow redirects like httpx
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

        // Disable SSL verification for local dev (Windows common issue)
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);

        // Cookie handling
        curl_setopt($ch, CURLOPT_COOKIEJAR, $this->cookieFile);
        curl_setopt($ch, CURLOPT_COOKIEFILE, $this->cookieFile);

        if ($body !== null) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body));
        }

        $response = curl_exec($ch);
        $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);

        curl_close($ch);

        if ($response === false) {
            throw new Exception("cURL connection failed: $error");
        }

        $data = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $data = $response;
        }

        return [$data, (int) $status];
    }

    private function uploadToS3($uploadUrl, $imgData)
    {
        $ch = curl_init();

        $headers = [
            "Content-Type: image/png",
            "Content-Length: " . strlen($imgData)
        ];

        curl_setopt($ch, CURLOPT_URL, $uploadUrl);
        curl_setopt($ch, CURLOPT_PUT, true); // Use PUT
        curl_setopt($ch, CURLOPT_INFILE, $fp = fopen('php://memory', 'r+'));
        fwrite($fp, $imgData);
        rewind($fp);
        curl_setopt($ch, CURLOPT_INFILESIZE, strlen($imgData));

        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

        // Disable SSL verification for S3 as well
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);

        // S3 generally doesn't need the session cookies, but it doesn't hurt.
        // Actually, S3 might reject "Cookie" headers if they are for SheerID domain? 
        // Best to NOT send SheerID cookies to S3.
        // So we do NOT load the cookie jar here.

        $response = curl_exec($ch);
        $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);

        curl_close($ch);
        fclose($fp);

        if ($response === false) {
            // Log error?
            return false;
        }

        return $status >= 200 && $status < 300;
    }

    private function generateImageNode($firstName, $lastName, $schoolId)
    {
        // Execute Node.js script
        $cmd = sprintf('node img_gen.js "%s" "%s" "%s"', $firstName, $lastName, $schoolId);

        // Create a temporary file to store output to avoid pipe issues with binary
        $tempFile = tempnam(sys_get_temp_dir(), 's_id_');

        // Redirect output to temp file
        // On Windows
        $fullCmd = $cmd . ' > "' . $tempFile . '"';

        exec($fullCmd, $output, $returnVar);

        if ($returnVar !== 0) {
            // Check if node is installed
            throw new Exception("Node.js image generation failed. (Exit Code: $returnVar)");
        }

        $imgData = file_get_contents($tempFile);
        unlink($tempFile);

        if (empty($imgData)) {
            throw new Exception("Generated image is empty.");
        }

        return $imgData;
    }

    public function verify($firstName = null, $lastName = null, $email = null, $birthDate = null, $schoolId = null, callable $logger = null)
    {
        $log = function ($msg) use ($logger) {
            if ($logger) {
                call_user_func($logger, $msg);
            } else {
                echo $msg . "\n";
            }
        };

        try {
            $currentStep = "initial";

            if (!$firstName || !$lastName) {
                $name = NameGenerator::generate();
                $firstName = $name['first_name'];
                $lastName = $name['last_name'];
            }

            $schoolId = $schoolId ?: Config::DEFAULT_SCHOOL_ID;
            $school = Config::SCHOOLS[$schoolId];

            if (!$email) {
                $email = generate_psu_email($firstName, $lastName);
            }
            if (!$birthDate) {
                $birthDate = generate_birth_date();
            }

            $log("Student Info: $firstName $lastName");
            $log("Email: $email");
            $log("School: {$school['name']}");
            $log("DOB: $birthDate");
            $log("Verification ID: {$this->verificationId}");

            // Step 1: Generate Image
            $log("Step 1/4: Generating Student ID PNG (via Node.js)...");
            $imgData = $this->generateImageNode($firstName, $lastName, $schoolId);
            $fileSize = strlen($imgData);
            $log("✅ PNG Size: " . round($fileSize / 1024, 2) . "KB");

            // Step 2: Submit Personal Info
            $log("Step 2/4: Submitting Personal Info...");
            $step2Body = [
                "firstName" => $firstName,
                "lastName" => $lastName,
                "birthDate" => $birthDate,
                "email" => $email,
                "phoneNumber" => "",
                "organization" => [
                    "id" => (int) $schoolId,
                    "idExtended" => $school["idExtended"],
                    "name" => $school["name"]
                ],
                "deviceFingerprintHash" => $this->deviceFingerprint,
                "locale" => "en-US",
                "metadata" => [
                    "marketConsentValue" => false,
                    "refererUrl" => Config::SHEERID_BASE_URL . "/verify/" . Config::PROGRAM_ID . "/?verificationId=" . $this->verificationId,
                    "verificationId" => $this->verificationId,
                    "flags" => '{"collect-info-step-email-first":"default","doc-upload-considerations":"default","doc-upload-may24":"default","doc-upload-redesign-use-legacy-message-keys":false,"docUpload-assertion-checklist":"default","font-size":"default","include-cvec-field-france-student":"not-labeled-optional"}',
                    "submissionOptIn" => "By submitting the personal information above, I acknowledge that my personal information is being collected under the privacy policy of the business from which I am seeking a discount"
                ]
            ];

            list($step2Data, $step2Status) = $this->sheeridRequest(
                "POST",
                Config::SHEERID_BASE_URL . "/rest/v2/verification/" . $this->verificationId . "/step/collectStudentPersonalInfo",
                $step2Body
            );

            if ($step2Status != 200) {
                // Return detailed error if available
                $err = is_array($step2Data) ? json_encode($step2Data) : $step2Data;
                throw new Exception("Step 2 failed (Status $step2Status): $err");
            }

            $log("✅ Step 2 Complete: " . ($step2Data['currentStep'] ?? 'unknown'));
            $currentStep = $step2Data['currentStep'] ?? $currentStep;

            // Step 3: Skip SSO if needed
            if (in_array($currentStep, ["sso", "collectStudentPersonalInfo"])) {
                $log("Step 3/4: Skipping SSO...");
                list($step3Data, $step3Status) = $this->sheeridRequest(
                    "DELETE",
                    Config::SHEERID_BASE_URL . "/rest/v2/verification/" . $this->verificationId . "/step/sso"
                );
                $log("✅ Step 3 Complete: " . ($step3Data['currentStep'] ?? 'unknown'));
                $currentStep = $step3Data['currentStep'] ?? $currentStep;
            }

            // Step 4: Upload Document
            $log("Step 4/4: Requesting upload URL...");
            $step4Body = [
                "files" => [
                    ["fileName" => "student_card.png", "mimeType" => "image/png", "fileSize" => $fileSize]
                ]
            ];

            list($step4Data, $step4Status) = $this->sheeridRequest(
                "POST",
                Config::SHEERID_BASE_URL . "/rest/v2/verification/" . $this->verificationId . "/step/docUpload",
                $step4Body
            );

            if (empty($step4Data['documents'])) {
                $err = is_array($step4Data) ? json_encode($step4Data) : $step4Data;
                throw new Exception("Failed to get upload URL: $err");
            }

            $uploadUrl = $step4Data['documents'][0]['uploadUrl'];
            $log("✅ Upload URL received.");

            if (!$this->uploadToS3($uploadUrl, $imgData)) {
                throw new Exception("S3 Upload Failed");
            }
            $log("✅ Image uploaded to S3.");

            list($step6Data, $step6Status) = $this->sheeridRequest(
                "POST",
                Config::SHEERID_BASE_URL . "/rest/v2/verification/" . $this->verificationId . "/step/completeDocUpload"
            );

            $log("✅ Submission Complete: " . ($step6Data['currentStep'] ?? 'unknown'));

            return [
                "success" => true,
                "pending" => true,
                "message" => "Docs submitted, pending review",
                "verification_id" => $this->verificationId,
                "redirect_url" => $step6Data['redirectUrl'] ?? null
            ];

        } catch (Exception $e) {
            $log("❌ Verification Failed: " . $e->getMessage());
            return ["success" => false, "message" => $e->getMessage()];
        }
    }
}
